use crate::{halt, serial_print, serial_println};
use core::{any::type_name, panic::PanicInfo};
use x86_64::instructions::port::Port;

//

#[derive(Debug, Clone, Copy, PartialEq, Eq, Default)]
#[repr(u32)]
pub enum QemuExitCode {
    #[default]
    Success = 0x10,
    Failed = 0x11,
}

//

pub trait TestCase {
    fn run(&self);
}

//

impl<T: Fn()> TestCase for T {
    fn run(&self) {
        serial_print!(" - {} ... ", type_name::<Self>());
        self();
        serial_println!("[ok]");
    }
}

//

#[panic_handler]
fn panic(info: &PanicInfo) -> ! {
    serial_println!("[failed]\n\nError: {info}\n");
    exit_qemu(QemuExitCode::Failed);
}

pub fn test_runner(tests: &[&dyn TestCase]) {
    serial_println!("Running {} tests", tests.len());
    for test in tests {
        test.run();
    }
    serial_println!("Success");
    exit_qemu(QemuExitCode::Success);
}

pub fn exit_qemu(exit_code: QemuExitCode) -> ! {
    unsafe {
        let mut port = Port::new(0xf4);
        port.write(exit_code as u32);
    }
    halt()
}
