/*
 * Copyright 2019 Nicolas Pope
 */

#ifndef _FTL_CALIBRATION_HPP_
#define _FTL_CALIBRATION_HPP_

#include <opencv2/core.hpp>
#include <opencv2/core/cuda.hpp>

#include <string>
#include <vector>

#include <ftl/codecs/channels.hpp>
#include <ftl/rgbd/camera.hpp>
#include <ftl/calibration/structures.hpp>

namespace ftl {
namespace rgbd {
namespace detail {

/**
 * Stereo rectification. Performs rectification for left and right channels.
 * Rectified image is same size as input image. Camera parameters reported by
 * getPose() and cameraMatrix() are for rectified camera (if enabled and valid
 * calibration set).
 */
class StereoRectification : public ftl::Configurable {
public:
	StereoRectification(nlohmann::json &config, cv::Size image_size);

	/** Set OpenCV interpolation mode, see cv::InterpolationFlags.
	 * NOTE: Artifacts possible if modified and rectify() is called in another
	 * thread (no synchronization)
	*/
	void setInterpolation(int interpolation);

	/**
	 * Calculate rectification parameters from given calibration.
	 */
	void setCalibration(ftl::calibration::CalibrationData &calib);
	bool calibrated();

	void rectify(cv::InputArray im, cv::OutputArray out, ftl::codecs::Channel c);

	/** Enable/disable rectification. TODO: move outside (to stereovideo)?
	 */
	void setEnabled(bool enabled);
	bool enabled();

	/** Get camera pose (camera to world). Returns rectified pose if
	 * rectification is enabled (and valid calibration is set).
	 */
	cv::Mat getPose(ftl::codecs::Channel c = ftl::codecs::Channel::Left);

	/** Get intrinsic matrix. */
	cv::Mat cameraMatrix(ftl::codecs::Channel c = ftl::codecs::Channel::Left);
	cv::Mat cameraMatrix(cv::Size size, ftl::codecs::Channel c = ftl::codecs::Channel::Left);

	/** Stereo baseline. In same unit as calibration (usually meters) */
	double baseline();
	/** Disparity offset (pixels) */
	double doff();
	double doff(cv::Size);

protected:
	void updateCalibration_();   // update calibration and calculate new params
	void calculateParameters_(); // re-calculate rectification maps and params

private:
	ftl::calibration::CalibrationData::Calibration calib_left_;
	ftl::calibration::CalibrationData::Calibration calib_right_;

	cv::Size image_resolution_;

	// rectification parameters
	bool enabled_;
	bool valid_;
	int interpolation_;
	double baseline_;
	cv::Mat R_; // rotation left to right
	cv::Mat t_; // translation left to right
	cv::Mat Q_;
	cv::Mat R_l_;
	cv::Mat R_r_;
	cv::Mat P_l_;
	cv::Mat P_r_;

	// rectification maps for cv::remap(); should be CV_16SC2 if remap done on
	// CPU and CV_32SC2 for GPU (generated by calculateParameters(), used by
	// rectify())
	// https://docs.opencv.org/master/da/d54/group__imgproc__transform.html
	int map_format_ = CV_16SC2;
	std::pair<cv::Mat,cv::Mat> map_l_;
	std::pair<cv::Mat,cv::Mat> map_r_;

	// temporary buffers for left/right
	cv::Mat tmp_l_;
	cv::Mat tmp_r_;
};

}
}
}

#endif // _FTL_CALIBRATION_HPP_

