#include <ftl/streams/renderer.hpp>
#include <ftl/rgbd/frame.hpp>
#include <ftl/rgbd/frameset.hpp>
#include <ftl/rgbd/capabilities.hpp>
#include <ftl/operators/antialiasing.hpp>
#include <ftl/operators/gt_analysis.hpp>
#include <ftl/utility/matrix_conversion.hpp>

#include <loguru.hpp>

#include "screen_render.hpp"
#include "collisions.hpp"

using ftl::render::Source;
using ftl::render::ScreenRender;
using ftl::codecs::Channel;
using ftl::rgbd::Capability;


ScreenRender::ScreenRender(ftl::render::Source *host, ftl::stream::Feed *feed)
: ftl::render::BaseSourceImpl(host), feed_(feed), my_id_(0), post_pipe_(nullptr) {
	/*host->restore("device:render", {
		"renderer",
		"source",
		"intrinsics",
		"name"
	});*/

	renderer_ = std::unique_ptr<ftl::render::CUDARender>(
		ftl::create<ftl::render::CUDARender>(host_, "renderer")
	);

	intrinsics_ = ftl::create<ftl::Configurable>(host_, "intrinsics");
	refresh_calibration_ = false;

	intrinsics_->onAny({"focal","width","height"}, [this]() {
		refresh_calibration_ = true;
	});

	filter_ = nullptr;
	std::string source = host_->value("source", std::string(""));

	if (source.size() > 0) {
		filter_ = feed_->filter({source},{Channel::Colour, Channel::Depth});
	} else {
		filter_ = feed_->filter({Channel::Colour, Channel::Depth});
	}

	host_->on("source", [this]() {
		std::string source = host_->value("source", std::string(""));

		if (source.size() > 0) {
			if (filter_) filter_->remove();
			filter_ = feed_->filter({source},{Channel::Colour, Channel::Depth});
		} else {
			if (filter_) filter_->remove();
			filter_ = feed_->filter({Channel::Colour, Channel::Depth});
		}
	});
}

ScreenRender::~ScreenRender() {
	if (filter_) filter_->remove();
	delete intrinsics_;
	if (post_pipe_) delete post_pipe_;
}

bool ScreenRender::isReady() {
	return true;
}

bool ScreenRender::capture(int64_t ts) {
	return true;
}

bool ScreenRender::retrieve(ftl::data::Frame &frame_out) {
	//auto input = std::atomic_load(&input_);

	my_id_ = frame_out.frameset();

	auto sets = filter_->getLatestFrameSets();

	if (sets.size() > 0) {
		ftl::rgbd::Frame &rgbdframe = frame_out.cast<ftl::rgbd::Frame>();

		if (!frame_out.has(Channel::Calibration) || refresh_calibration_) {
			refresh_calibration_ = false;
			rgbdframe.setLeft() = ftl::rgbd::Camera::from(intrinsics_);

			if (!frame_out.has(Channel::Capabilities)) {
				auto &cap = frame_out.create<std::unordered_set<Capability>>(Channel::Capabilities);
				cap.emplace(Capability::VIDEO);
				cap.emplace(Capability::MOVABLE);
				cap.emplace(Capability::ADJUSTABLE);
				cap.emplace(Capability::VIRTUAL);
				cap.emplace(Capability::LIVE);
			}

			auto &meta = frame_out.create<std::map<std::string,std::string>>(Channel::MetaData);
			meta["name"] = host_->value("name", host_->getID());
			meta["id"] = host_->getID();
			meta["uri"] = std::string("device:render");
			meta["device"] = std::string("CUDA Render");
		}
		if (!frame_out.has(Channel::Pose)) {
			rgbdframe.setPose() = Eigen::Matrix4d::Identity();
		}

		int width = rgbdframe.getLeft().width;
		int height = rgbdframe.getLeft().height;
		
		// FIXME: Create opengl buffers here and map to cuda?
		rgbdframe.create<cv::cuda::GpuMat>(Channel::Colour).create(height, width, CV_8UC4);
		rgbdframe.create<cv::cuda::GpuMat>(Channel::Depth).create(height, width, CV_32F);
		rgbdframe.createTexture<float>(Channel::Depth);

		try {
			renderer_->begin(rgbdframe, ftl::codecs::Channel::Left);

			for (auto &s : sets) {
				if (s->frameset() == my_id_) continue;  // Skip self

				renderer_->submit(
					s.get(),
					ftl::codecs::Channels<0>(ftl::codecs::Channel::Colour),
					Eigen::Matrix4d::Identity());

				// TODO: Render audio also...
				// Use another thread to merge all audio channels along with
				// some degree of volume adjustment. Later, do 3D audio.
			}

			renderer_->render();

			renderer_->end();
		} catch (const ftl::exception &e) {
			LOG(ERROR) << "Render exception: " << e.what();
		}

		if (!post_pipe_) {
			post_pipe_ = ftl::config::create<ftl::operators::Graph>(host(), "post_filters");
			post_pipe_->append<ftl::operators::FXAA>("fxaa");
			post_pipe_->append<ftl::operators::GTAnalysis>("gtanalyse");
		}

		post_pipe_->apply(rgbdframe, rgbdframe, 0);

		if (host_->value("enable_touch", false)) {
			ftl::render::collision2touch(rgbdframe, renderer_->getCollisions(), sets, my_id_, host_->value("touch_min", 0.01f), host_->value("touch_max", 0.05f));
		}
		return true;
	} else {
		//LOG(INFO) << "Render fail";
		return true;
	}
}
