#include <ftl/streams/builder.hpp>
#include <ftl/timer.hpp>

#define LOGURU_REPLACE_GLOG 1
#include <loguru.hpp>

#include <chrono>

using ftl::streams::BaseBuilder;
using ftl::streams::ForeignBuilder;
using ftl::streams::LocalBuilder;
using ftl::streams::IntervalSourceBuilder;
using ftl::streams::ManualSourceBuilder;
using ftl::streams::LockedFrameSet;
using ftl::data::FrameSet;
using ftl::data::Frame;
using namespace std::chrono;
using std::this_thread::sleep_for;


/*float Builder::latency__ = 0.0f;
float Builder::fps__ = 0.0f;
int Builder::stats_count__ = 0;
MUTEX Builder::msg_mutex__;*/

BaseBuilder::BaseBuilder(ftl::data::Pool *pool, int id) : pool_(pool), id_(id) {
	size_ = 1;
}

BaseBuilder::BaseBuilder() : pool_(nullptr), id_(0) {
	size_ = 1;
}

BaseBuilder::~BaseBuilder() {

}

// =============================================================================

LocalBuilder::LocalBuilder(ftl::data::Pool *pool, int id) : BaseBuilder(pool, id) {
	// Host receives responses that must propagate
	ctype_ = ftl::data::ChangeType::FOREIGN;
}

LocalBuilder::LocalBuilder() : BaseBuilder() {
	// Host receives responses that must propagate
	ctype_ = ftl::data::ChangeType::FOREIGN;
}

LocalBuilder::~LocalBuilder() {

}

LockedFrameSet LocalBuilder::get(int64_t timestamp, size_t ix) {
	SHARED_LOCK(mtx_, lk);
	if (!frameset_) {
		frameset_ = _allocate(timestamp);
	}

	LockedFrameSet lfs(frameset_.get(), frameset_->smtx);
	return lfs;
}

LockedFrameSet LocalBuilder::get(int64_t timestamp) {
	SHARED_LOCK(mtx_, lk);
	if (!frameset_) {
		frameset_ = _allocate(timestamp);
	}

	LockedFrameSet lfs(frameset_.get(), frameset_->smtx);
	return lfs;
}

void LocalBuilder::setFrameCount(size_t size) {
	// TODO: Resize the buffered frame!?
	size_ = size;
}

std::shared_ptr<ftl::data::FrameSet> LocalBuilder::getNextFrameSet(int64_t ts) {
	UNIQUE_LOCK(mtx_, lk);
	if (!frameset_) {
		frameset_ = _allocate(ts);
	}
	auto fs = frameset_;
	frameset_ = _allocate(ts+1);
	lk.unlock();

	// Must lock to ensure no updates can happen here
	UNIQUE_LOCK(fs->smtx, lk2);
	fs->changeTimestamp(ts);
	fs->store();
	//for (auto &f : fs->frames) {
	//	f.store();
	//}
	return fs;
}

std::shared_ptr<ftl::data::FrameSet> LocalBuilder::_allocate(int64_t timestamp) {
	auto newf = std::make_shared<FrameSet>(pool_, ftl::data::FrameID(id_,255), timestamp);
	for (size_t i=0; i<size_; ++i) {
		newf->frames.push_back(std::move(pool_->allocate(ftl::data::FrameID(id_, i), timestamp)));
	}

	newf->count = size_;
	newf->mask = 0xFF;
	newf->clearFlags();
	newf->pose.setIdentity();
	return newf;
}

// =============================================================================

IntervalSourceBuilder::IntervalSourceBuilder(ftl::data::Pool *pool, int id, ftl::data::DiscreteSource *src) : LocalBuilder(pool, id), srcs_({src}) {

}

IntervalSourceBuilder::IntervalSourceBuilder(ftl::data::Pool *pool, int id, const std::list<ftl::data::DiscreteSource*> &srcs) : LocalBuilder(pool, id), srcs_(srcs) {

}

IntervalSourceBuilder::IntervalSourceBuilder() : LocalBuilder() {

}

IntervalSourceBuilder::~IntervalSourceBuilder() {

}

void IntervalSourceBuilder::start() {
	capture_ = std::move(ftl::timer::add(ftl::timer::timerlevel_t::kTimerHighPrecision, [this](int64_t ts) {
		for (auto *s : srcs_) s->capture(ts);
		return true;
	}));

	retrieve_ = std::move(ftl::timer::add(ftl::timer::timerlevel_t::kTimerMain, [this](int64_t ts) {
		auto fs = getNextFrameSet(ts);

		// TODO: Do in parallel...
		for (auto *s : srcs_) {
			if (!s->retrieve(fs->firstFrame())) {
				LOG(WARNING) << "Frame is being skipped: " << ts;
				fs->firstFrame().warning("Frame is being skipped");
			}
		}

		cb_.trigger(fs);
		return true;
	}));
}

void IntervalSourceBuilder::stop() {
	capture_.cancel();
	retrieve_.cancel();
}

// =============================================================================

ManualSourceBuilder::ManualSourceBuilder(ftl::data::Pool *pool, int id, ftl::data::DiscreteSource *src) : LocalBuilder(pool, id), src_(src) {

}

ManualSourceBuilder::ManualSourceBuilder() : LocalBuilder(), src_(nullptr) {

}

ManualSourceBuilder::~ManualSourceBuilder() {

}

void ManualSourceBuilder::tick() {
	if (!src_) return;

	int64_t ts = ftl::timer::get_time();
	if (ts < last_timestamp_ + mspf_) return;
	last_timestamp_ = ts;

	src_->capture(ts);

	auto fs = getNextFrameSet(ts);

	if (!src_->retrieve(fs->firstFrame())) {
		LOG(WARNING) << "Frame was skipping";
	}

	cb_.trigger(fs);
}

// =============================================================================

ForeignBuilder::ForeignBuilder(ftl::data::Pool *pool, int id) : BaseBuilder(pool, id), head_(0) {
	jobs_ = 0;
	skip_ = false;
	bufferSize_ = 1;
	last_frame_ = 0;

	mspf_ = ftl::timer::getInterval();
	name_ = "NoName";
}

ForeignBuilder::ForeignBuilder() : BaseBuilder(), head_(0) {
	jobs_ = 0;
	skip_ = false;
	bufferSize_ = 1;
	last_frame_ = 0;

	mspf_ = ftl::timer::getInterval();
	name_ = "NoName";
}

ForeignBuilder::~ForeignBuilder() {
	main_id_.cancel();

	UNIQUE_LOCK(mutex_, lk);
	// Make sure all jobs have finished
	while (jobs_ > 0) {
		sleep_for(milliseconds(10));
	}
}

LockedFrameSet ForeignBuilder::get(int64_t timestamp) {
	if (timestamp <= 0) throw FTL_Error("Invalid frame timestamp");

	UNIQUE_LOCK(mutex_, lk);

	auto fs = _get(timestamp);
	LockedFrameSet lfs(fs.get(), fs->smtx, [this,fs](ftl::data::FrameSet *d) {
		if (fs->isComplete()) {
			if (bufferSize_ == 0 && !fs->test(ftl::data::FSFlag::STALE)) {
				UNIQUE_LOCK(mutex_, lk);
				_schedule();
			}
		}
	});
	return lfs;
}

std::shared_ptr<ftl::data::FrameSet> ForeignBuilder::_get(int64_t timestamp) {
	if (timestamp <= last_frame_) {
		throw FTL_Error("Frameset already completed: " << timestamp);
	}

	auto fs = _findFrameset(timestamp);

	if (!fs) {
		// Add new frameset
		fs = _addFrameset(timestamp);
		if (!fs) throw FTL_Error("Could not add frameset");

		_schedule();
	}

	if (fs->test(ftl::data::FSFlag::STALE)) {
		throw FTL_Error("Frameset already completed");
	}
	return fs;
}

LockedFrameSet ForeignBuilder::get(int64_t timestamp, size_t ix) {
	if (ix == 255) {
		UNIQUE_LOCK(mutex_, lk);

		if (timestamp <= 0) throw FTL_Error("Invalid frame timestamp (" << timestamp << ")");
		auto fs = _get(timestamp);
		if (!fs) throw FTL_Error("No frameset for time " << timestamp);

		LockedFrameSet lfs(fs.get(), fs->smtx);
		return lfs;
	} else {
		if (timestamp <= 0 || ix >= 32) throw FTL_Error("Invalid frame timestamp or index (" << timestamp << ", " << ix << ")");

		UNIQUE_LOCK(mutex_, lk);

		if (ix >= size_) {
			size_ = ix+1;
		}

		auto fs = _get(timestamp);

		if (ix >= fs->frames.size()) {
			//throw FTL_Error("Frame index out-of-bounds - " << ix << "(" << fs->frames.size() << ")");

			// FIXME: This is really dangerous
			while (fs->frames.size() < size_) {
				fs->frames.push_back(std::move(pool_->allocate(ftl::data::FrameID(fs->frameset(), + fs->frames.size()), fs->timestamp())));
			}
		}

		LockedFrameSet lfs(fs.get(), fs->smtx, [this,fs](ftl::data::FrameSet *d) {
			// TODO: schedule completed
			if (fs->isComplete()) {
				if (bufferSize_ == 0 && !fs->test(ftl::data::FSFlag::STALE)) {
					UNIQUE_LOCK(mutex_, lk);
					_schedule();
				}
			}
		});

		return lfs;
	}
}

/*void Builder::completed(int64_t ts, size_t ix) {
	std::shared_ptr<ftl::data::FrameSet> fs;

	{
		UNIQUE_LOCK(mutex_, lk);
		fs = _findFrameset(ts);
	}

	if (fs && ix == 255) {
		// Note: Frameset can't complete without frames
		//if (bufferSize_ == 0 && !fs->test(ftl::data::FSFlag::STALE) && static_cast<unsigned int>(fs->count) >= size_) {
		//	UNIQUE_LOCK(mutex_, lk);
		//	_schedule();
		//}
	} else if (fs && ix < fs->frames.size()) {
		{
			UNIQUE_LOCK(fs->mutex(), lk2);

			// If already completed for given frame, then skip
			if (fs->mask & (1 << ix)) return;

			//states_[ix] = fs->frames[ix].origin();
			fs->mask |= (1 << ix);
			++fs->count;
		}

		//LOG(INFO) << "COMPLETE FRAME : " << fs->timestamp() << " " << fs->count << "(" << size_ << ")";

		// No buffering, so do a schedule here for immediate effect
		if (bufferSize_ == 0 && !fs->test(ftl::data::FSFlag::STALE) && static_cast<unsigned int>(fs->count) >= size_) {
			UNIQUE_LOCK(mutex_, lk);
			_schedule();
		}
	} else {
		LOG(ERROR) << "Completing frame that does not exist: " << ts << ":" << ix;
	}
}*/

/*void Builder::markPartial(int64_t ts) {
	std::shared_ptr<ftl::data::FrameSet> fs;

	{
		UNIQUE_LOCK(mutex_, lk);
		fs = _findFrameset(ts);
		if (fs) fs->set(ftl::data::FSFlag::PARTIAL);
	}
}*/

void ForeignBuilder::_schedule() {
	if (size_ == 0) return;
	std::shared_ptr<ftl::data::FrameSet> fs;

	// Still working on a previously scheduled frame
	if (jobs_ > 0) return;

	// Find a valid / completed frameset to process
	fs = _getFrameset();

	// We have a frameset so create a thread job to call the onFrameset callback
	if (fs) {
		jobs_++;

		ftl::pool.push([this,fs](int) {
			// Calling onFrameset but without all frames so mark as partial
			if (static_cast<size_t>(fs->count) < fs->frames.size()) fs->set(ftl::data::FSFlag::PARTIAL);

			//for (auto &f : fs->frames) f.store();
			fs->store();

			//UNIQUE_LOCK(fs->mutex(), lk2);

			try {
				cb_.trigger(fs);
			} catch(const ftl::exception &e) {
				LOG(ERROR) << "Exception in frameset builder: " << e.what();
				LOG(ERROR) << "Trace = " << e.trace();
			} catch(std::exception &e) {
				LOG(ERROR) << "Exception in frameset builder: " << e.what();
			}

			UNIQUE_LOCK(mutex_, lk);
			//_freeFrameset(fs);
			jobs_--;

			// Schedule another frame immediately (or try to)
			_schedule();
		});
	}

}

static void mergeFrameset(ftl::data::FrameSet &f1, ftl::data::FrameSet &f2) {
	// Prepend all frame encodings in f2 into corresponding frame in f1.
	/*for (size_t i=0; i<f1.frames.size(); ++i) {
		if (f2.frames.size() <= i) break;
		f1.frames[i].mergeEncoding(f2.frames[i]);
	}*/
}

/*void Builder::_recordStats(float fps, float latency) {
	UNIQUE_LOCK(msg_mutex__, lk);
	latency__ += latency;
	fps__ += fps;
	++stats_count__;
}*/

std::pair<float,float> BaseBuilder::getStatistics() {
	/*UNIQUE_LOCK(msg_mutex__, lk);
	if (stats_count__ == 0.0f) return {0.0f,0.0f};
	fps__ /= float(stats_count__);
	latency__ /= float(stats_count__);
	float fps = fps__;
	float latency = latency__;
	//LOG(INFO) << name_ << ": fps = " << fps_ << ", latency = " << latency_;
	fps__ = 0.0f;
	latency__ = 0.0f;
	stats_count__ = 0;*/
	return {-1.0f, -1.0f};
}

std::shared_ptr<ftl::data::FrameSet> ForeignBuilder::_findFrameset(int64_t ts) {
	// Search backwards to find match
	for (auto f : framesets_) {
		if (f->timestamp() == ts) {
			return f;
		} else if (f->timestamp() < ts) {
			return nullptr;
		}
	}

	return nullptr;
}

/*
 * Get the most recent completed frameset that isn't stale.
 * Note: Must occur inside a mutex lock.
 */
std::shared_ptr<ftl::data::FrameSet> ForeignBuilder::_getFrameset() {
	//LOG(INFO) << "BUF SIZE = " << framesets_.size();

	auto i = framesets_.begin();
	int N = bufferSize_;

	// Skip N frames to fixed buffer location
	if (bufferSize_ > 0) {
		while (N-- > 0 && i != framesets_.end()) ++i;
	// Otherwise skip to first fully completed frame
	} else {
		while (i != framesets_.end() && static_cast<size_t>((*i)->count) < (*i)->frames.size()) ++i;
	}

	if (i != framesets_.end()) {
		auto f = *i;
		last_frame_ = f->timestamp();

		// Lock to force completion of on going construction first
		UNIQUE_LOCK(f->smtx, slk);
		auto j = framesets_.erase(i);
		f->set(ftl::data::FSFlag::STALE);
		slk.unlock();

		int count = 0;
		// Merge all previous frames
		// TODO: Remove?
		while (j!=framesets_.end()) {
			++count;

			auto f2 = *j;
			LOG(WARNING) << "FrameSet discarded: " << f2->timestamp();
			{
				// Ensure frame processing is finished first
				UNIQUE_LOCK(f2->smtx, lk);
				j = framesets_.erase(j);
			}
			mergeFrameset(*f,*f2);

			//LOG(INFO) << "EXTRA REMOVED";
			//_freeFrameset(f2);
		}

		//if (count > 0) LOG(INFO) << "COUNT = " << count;

		int64_t now = ftl::timer::get_time();
		//float framerate = 1000.0f / float(now - last_ts_);
		//_recordStats(framerate, now - f->timestamp());
		last_ts_ = now;
		return f;
	}

	return nullptr;
}

std::shared_ptr<ftl::data::FrameSet> ForeignBuilder::_addFrameset(int64_t timestamp) {
	if (framesets_.size() >= 32) {
		LOG(WARNING) << "Frameset buffer full, resetting: " << timestamp;
		framesets_.clear();
	}

	auto newf = std::make_shared<FrameSet>(pool_, ftl::data::FrameID(id_,255), timestamp);
	for (size_t i=0; i<size_; ++i) {
		newf->frames.push_back(std::move(pool_->allocate(ftl::data::FrameID(id_, i), timestamp)));
	}

	newf->count = 0;
	newf->mask = 0;
	newf->clearFlags();
	//newf->frames.resize(size_);
	newf->pose.setIdentity();
	//newf->clearData();

	// Insertion sort by timestamp
	for (auto i=framesets_.begin(); i!=framesets_.end(); i++) {
		auto f = *i;

		if (timestamp > f->timestamp()) {
			framesets_.insert(i, newf);
			return newf;
		}
	}

	framesets_.push_back(newf);
	return newf;
}

/*void Builder::setName(const std::string &name) {
	name_ = name;
}*/
