#!/usr/bin/env python
# coding=utf-8
"""
Created on Tue Feb 20 21:33:57 2018

@author: veskuh
"""

import os, sys, cv2, requests, logging, re
from requests_toolbelt.multipart.encoder import MultipartEncoder
from datetime import datetime

# Image file name and location
TMP_IMAGE = os.path.join(os.path.normpath(os.path.join(os.getcwd(), 'images')), 'image_{0}.jpg'.format(datetime.now().strftime("%Y%m%d%H%M%S")))
# Face detection model path
MODEL_PATH = os.path.join(os.path.normpath(os.path.join(os.getcwd(), 'haarcascades')), 'haarcascade_frontalface_default.xml')
# Server URI
SERVER_URL = "http://tensorflow.stop.capstone.utu.fi"

# logging.basicConfig(filename = os.path.normpath(os.path.join(os.getcwd(), 'debug.log')), level = logging.DEBUG)
logging.basicConfig(filename = os.path.normpath(os.path.join(os.getcwd(), 'error.log')), level = logging.ERROR)

# Sending image to server
def sendImage(stopCode = None):
    # Send local image file to tensorflow-client
    stopCode = stopCode if stopCode else "unknown"
    try:
        with open(TMP_IMAGE, "rb") as fp:
            multipart_data = MultipartEncoder(
                fields = {
                    "stop_code": stopCode,
                    "image": (os.path.basename(TMP_IMAGE), fp, "image/jpeg")
               }
            )

            response = requests.post(SERVER_URL + "/api/v1/classifyImage/",
                data = multipart_data,
                headers = {"Content-Type": multipart_data.content_type}
            )

            if response.status_code != 200:
                raise Exception('Service unavailable')

    except Exception as err:
        logError('An error has occurred while sending the file: "{0}", for: {1}'.format(err, stopCode))

# Removing image for security purposes 
def removeImage():
    try:
        os.remove(TMP_IMAGE)
    except Exception as err:
        logError('An error has occurred while deleting temporary file: "{0}"'.format(err))

# Taking a picture from default webcam and saving it
def takePicture(cameraIndex, model):
    try:
        cap = cv2.VideoCapture(cameraIndex)
        if not cap.isOpened():
            raise Exception('Camera not available')

        ret, frame = cap.read()
        if not ret:
            raise Exception('Unable to capture image')

        # matching serverside parameters, for comparable results
        faces = detectFaces(model, cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY))

        if len(faces) != 0:
            cv2.imwrite(TMP_IMAGE, frame)

        cap.release()

        if not os.path.isfile(TMP_IMAGE):
            raise Exception('Image not saved, no faces in image')

        return True
    except Exception as err:
        logError('An error has occurred while preprocessing camera image: "{0}"'.format(err))
        return False

def logError(msg):
    logging.error('{0}:{1}'.format(datetime.now().strftime("%Y-%m-%d %H:%M:%S"), msg))

def detectFaces(model, image):
    return model.detectMultiScale(image,
        scaleFactor=1.1,
        minNeighbors=5,
        minSize=(30, 30),
        flags=cv2.CASCADE_SCALE_IMAGE
    )

def main(argv):
    if argv != None and (len(argv) <= 2 or not re.match("^[\w\d_-]+$", argv[1]) or not re.match("^[\d]+$", argv[2])):
        return False

    try:
        model = cv2.CascadeClassifier(MODEL_PATH)
        if takePicture(int(argv[2]), model):
            sendImage(str(argv[1]))
            removeImage()
    except Exception as err:
        logError('Invalid arguments provided for script "{0}"'.format(err))
        return False

    return True

if __name__ == "__main__":
    if not main(sys.argv):
        print("Stop identifier or video source not provided (missing argument/s)")
        exit(1)
